/***************************************************************************
 *
 * Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#ifndef _GAL2DGRAPHICSYSTEM_H_
#define _GAL2DGRAPHICSYSTEM_H_

#include "Configuration.h"
#include "HAL/gc_hal.h"
#include "GraphicSystems/BaseGraphicSystem.h"
#include "Log.h"

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pthread.h>

#include "wayland-util.h"

extern "C" {
#include <xf86drm.h>
#include <xf86drmMode.h>
#include "ipu_dp_csc.h"
}
#define MAX_TIME_TO_NEXT_PAGE_FLIP 1000

#define MAX_DUALPLAIN_CRTCS 2

#define ALPHA_MAX 255

#define GAMMA_ARRAY_SIZE 17

#define MAX_TILING_LEVEL 4

struct ConnectorCrtcAssignment {
    drmModeConnectorPtr drmConnector;
    int                 preferredCrtcID;
    uint32_t            ilmDispID;
    bool                initialized;
};

struct DrmMode {
        uint32_t          flags;
        int32_t           width;
        int32_t           height;
        uint32_t          refresh;
        uint32_t          refresh_ns;
        drmModeModeInfo   modeInfo;
        wl_list           link;
};

struct drm_buffer {
    uint32_t width;
    uint32_t height;
    uint32_t stride;
    uint32_t size;
    uint32_t handle;
    void *pixels;
    uint32_t fb;
};

class Gal2dGraphicSystem;

struct DrmOutput {
        struct wl_list    link;
        struct DrmMode*   currentMode;
        struct wl_list    modeList;
        uint32_t          crtcID;
        int               fdDev;
        uint32_t          connectorID;
        uint32_t connector_type;
        uint32_t connector_type_id;
        drmModeCrtcPtr    orgCrtc;
        bool               pageFlipPending;
        uint32_t          screenID;
        bool              isScreenshot;

        struct drm_buffer *buffer;
        gcoSURF*          dstSurfaces;
        gcoSURF           currentTargetSurface;
        gctUINT32         dstNumSurfaces;
        gceSURF_FORMAT    surfaceFormat;
        gctUINT32         currentIndex;
        BaseWindowSystem *windowSystem;
        Gal2dGraphicSystem *graphicSystem;
        bool              dither;
        uint32_t          depth;
        struct timespec    time_pageflip;
        drmModePropertyPtr propDPMS;
        int                dpms_current_state;
        int                dpms_readytoChange;
        int                dpms_newState;
        CLRPROP            clrprop;
        CLRSPACETYPE       type;
        double             screen_alpha;
        int                repaintWindowPeriodMS;
};

class IlmMatrix;
class FileThreadObject;

struct MultiSurfaceRegion
{
    FloatRectangle m_rect;
    SurfaceList m_surfaces;
};

struct MultiSrcSurface
{
	SurfaceList m_surfaces;
	bool supported;
	bool has_multi_planar_yuv;
};

struct VivTile
{
	unsigned int prevTileW;
	unsigned int prevTileH;
	unsigned int tileW;
	unsigned int tileH;
	unsigned int nextTileW;
	unsigned int nextTileH;
	unsigned int tileGroupDim;
};

class Gal2dGraphicSystem: public BaseGraphicSystem<void*, void*>
{
    friend class FileThreadObject;
public:
    Gal2dGraphicSystem(int windowHeight, int windowWidth,Configuration* mConfiguration);
    virtual ~Gal2dGraphicSystem();

    virtual bool init(void* display, void* window);

    virtual void clearBackground();

    virtual void activateGraphicContext();
    virtual void releaseGraphicContext(){}

    virtual void swapBuffers();

    virtual void beginLayer(Layer* layer);
    virtual void endLayer();

    virtual bool needsRedraw(Layer *layer);
    virtual bool needsRedraw(LayerList layers);
    virtual void renderSWLayer(Layer* layer, bool clear);
    virtual void renderSWLayers(LayerList layers, bool clear);
    virtual void startRepaintLoop(uint screenID);

    virtual bool initGal2d(int displayWidth, int displayHeight);
    virtual void resize(int displayWidth, int displayHeight);

    virtual void saveScreenShotOfFramebuffer(const std::string& fileToSave,
                                             DumpOutputType outputType);
    virtual void saveScreenShotOfLayer(const std::string& fileToSave,
                                       Layer* layer);
    virtual void saveScreenShotOfSurface(const std::string& fileToSave,
                                         Surface* surface);
    virtual void dumpSurfaceToFile(const std::string& fileToSave,
                                   Surface* surface);
    virtual bool getSurfaceIsRGB(Surface* surface, bool& isRGB);

    virtual unsigned int* getScreenIDs(unsigned int *nScreens);
    virtual unsigned int* getScreenResolution(uint screenID);
    virtual void getScreenResolution(int screenID, int &width, int &height);
    virtual bool createVncbuffer( int screenID,int bpp,void **pp_vncsurfptr,void **pp_buffaddr);
    virtual bool dumpFramebuffertovncsurf(int screenID);
    virtual bool destroyVncbuffer(int screenID,void *p_vncsurfptr);
    virtual bool setOptimizationMode(OptimizationType id, OptimizationModeType mode)
    {
        (void)id;
        (void)mode;
        return true;
    }
    virtual bool getOptimizationMode(OptimizationType id, OptimizationModeType *mode)
    {
        (void)id;
        (void)mode;
        return true;
    }

    virtual void multiSrcBlit(SurfaceList surfaces);

    virtual void renderSurface(Surface* surface);

    virtual gcoHAL getGcoHal()
    {
        return m_gcoHal;
    }

    virtual gco2D getGco2d()
    {
        return m_gco2dEngine;
    }

    virtual void updateScreenList(LmScreenList& screenList);
    virtual bool switchScreen(uint screenid, bool screenshot = false);
    virtual bool isFlipPending(uint screenid);
    virtual bool synchronizedSurfacesDamaged(LayerList layers);
    virtual bool isSurfaceTransparentOn(Surface* surf, int x , int y);
    virtual bool isSurfaceDirty(unsigned int surfID);
    virtual bool isScreenFrozen(unsigned int screenID);
    virtual bool setGamma(unsigned int screenID, double value);
    virtual bool setCSC(unsigned int screenID, ilmCSCProperties* pclrprop);
    virtual bool setDisplayState(unsigned int screenId, ilmScreenState screenState);
    virtual bool getDisplayState(unsigned int screenId, ilmScreenState *screenState);
    virtual bool setDisplayAlpha(unsigned int screenId, unsigned int alphaValue);
    virtual bool getDisplayAlpha(unsigned int screenId, unsigned int *alphaValue);

protected:
    virtual std::list<MultiSrcSurface*> constructMultiSrcList(SurfaceList opaqueRegSurfList, int maxSurf);
    virtual std::list<MultiSurfaceRegion*> computeRegions(SurfaceList opaqueRegSurfList, bool clear);
    virtual std::list<Surface*> getOpaqueRegSurfList(LayerList layers);
    virtual void setTargetSurface(gcoSURF surface);
    virtual void renderRegion(MultiSurfaceRegion* region, bool blend);
    virtual bool renderSurfaces(SurfaceList surfaces, FloatRectangle targetDestination, bool blend);
    virtual bool canSkipClear();
    virtual bool useSkipClear(LayerList layers);
    virtual void releaseResources();
    virtual bool enableAlphaBlend(gcoSURF surface, float opacity, bool useSurfaceAlpha);
    virtual int getDepthFromGcoSurfFormat(gceSURF_FORMAT format);
    uint32_t getScreenIdFromConnectorType(uint32_t connector_type,uint32_t connector_type_id);
    drmModePropertyPtr getDRMProperty(int drmFd, drmModeConnector* connector, char *pName);

    bool getDPMSValue(ilmScreenState screenState, int& value);
    bool getILMScreenValue(ilmScreenState* screenState, int value);

    int m_windowWidth;
    int m_windowHeight;
    void* m_nativeDisplay;
    void* m_nativeWindow;

    int m_displayWidth;
    int m_displayHeight;

    Layer* m_currentLayer;

    OptimizationModeType m_optimizations[OPT_COUNT];

    gctBOOL m_hw2DPE20;
    gcoOS m_gcoOs;
    gcoHAL m_gcoHal;
    gco2D m_gco2dEngine;

    gctPOINTER m_localDisplayInfo;


private:
    struct wl_list m_outputList;
    struct wl_list m_screenshotList;

    sem_t*          m_drm_master_sem;
    int             m_fdDev;
    drmModeCrtcPtr* m_crtcs;
    int             m_crtcsNum;
    DrmOutput*      m_currentOutput;
    uint32_t        m_availableCrtcs;
    uint32_t        m_usedCrtcs;

    int             dpms_fdDev;
    FileThreadObject *m_currentScreenshotThread;
    pthread_t m_fileThread;
    Configuration* mConfiguration;
    sem_t m_workerSem;
    int m_dualplain_crtcs[MAX_DUALPLAIN_CRTCS];

    int m_maxMultiSrc;
    pthread_mutex_t m_graSystemMutex;

    static VivTile validTileSizes[MAX_TILING_LEVEL];
    DrmOutput* findScreenshotOutput(uint screenID);
    void startScreenshotThread(FileThreadObject *thread);
    void writeBMPFile(const char *filename, int width, int height,
                      gceSURF_FORMAT format, char* buffer);
    void writeFile(const char *filename, int size, char* buffer, gcoSURF *tempSurf);
    void writeLayerFile(const char *filename,
                        int layerWidth, int layerHeight,
                        int blankWidth, int blankHeight,
                        int bottomSize, int bufferDepth,
                        size_t start_offset, size_t display_row_width,
                        size_t layer_row_width, char* buffer);
    void flushAndCommit();
    void saveScreenShot();
    bool initializeSystem();
    bool createOutputs();
    int  createOutputForConnector(drmModeConnector* connector, uint32_t dispID, int prefered_ctrc);
    int  createOutputForScreenshot(DrmOutput *screen);
    int  createBuffer(struct drm_buffer*,int bpp);
    int  drmOutputAddMode(struct DrmOutput* output, drmModeModeInfo* info);
    int  drmQueryMastership();
    bool useFilterBlit(gceSURF_FORMAT format);
    bool isYUVFomat(gceSURF_FORMAT format);
    bool initializeDrmFD();
    void* dumpSurfacetovncsurf( unsigned int screenID,void *p_vncsurfptr);
    void sort_dualplain_crtcs(int);
    unsigned int* ReadPixelValue(unsigned int *src, unsigned int x, unsigned int y, unsigned int tilingLevel);
    int setDpmsUponStateChange(struct DrmOutput *output);
    static int onDpmsInput(int fd, uint32_t mask, void *data);
    static int onDrmInput(int fd, uint32_t mask, void *data);
    static void pageFlipHandler(int fd, unsigned int frame, unsigned int sec,
                                    unsigned int usec, void *data);
};

#endif /* _GAL2DGRAPHICSYSTEM_H_ */
